;------------------------------------------------------------------------------
;
; Title:			ds30 Loader for PIC12F and PIC16F
;
; File description:	Main firmwarefile
;
; Copyright: 		Copyright � 2011, Mikael Gustafsson
;
; Version:			1.0.1 April 2011
;
; Webpage: 			http://mrmackey.no-ip.org/elektronik/ds30loader/
;
; Thanks to:		Claudio Chiculita, this code is based on the Tiny PIC bootloader
;
; History:			1.0.1 Bugfix: BRG16 was broken
;						  Improvement: supports uart 2
;						  Improvement: send unknown command
;						  Improvement: PIC12 support
;						  Improvement: support new PIC16's that has no eeprom
;					1.0.0 Improvement: improved timeout calculation
;					      Improvement: higher baudrates setting
;					      New feature: auto baud rate detection
;						  Bugfix: tx enable tris bit was not cleared
;					0.9.4 Improvement: New setting, HELLORETRIES
;					      Bugfix: Banksel added for RCREG to be compatible with some devices including 1827;					
;					0.9.3 added tx enable code
;						  compatible with pic16 with extended instruction set
;					0.9.2 -
;					0.9.1 Moved goto to usr appl. to make room for page selection
;					0.9.0 Initial release
;                                                                             
;------------------------------------------------------------------------------

;-----------------------------------------------------------------------------
;    This file is part of ds30 Loader.
;
;    ds30 Loader is free software: you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation.
;
;    ds30 Loader is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with ds30 Loader. If not, see <http://www.gnu.org/licenses/>.
;------------------------------------------------------------------------------


;------------------------------------------------------------------------------
; Includes
;------------------------------------------------------------------------------ 
		#include "settings.inc"
		
		
;------------------------------------------------------------------------------
; Defines
;------------------------------------------------------------------------------
		radix DEC
		
		#define	VERMAJ		1										;firmware version major
		#define	VERMIN		0										;firmware version minor
		#define	VERREV		1										;firmware version revision
		
		#define HELLO 		0xC1
		#define OK 			'K'										;erase/write ok
		#define CHECKSUMERR	'N'										;checksum error
		#define	VERFAIL		'V'										;verification failed
		#define	BLPROT     	'P'                              		;bl protection tripped
		#define	UCMD     	'U'                              		;unknown command	
				
		if BLTIME >= 500
			#define delay2		100
		else
			#define delay2		5
		endif		
		#define	BLDELAY		( BLTIME * (FOSC / 4000) / ((delay2 * ((256*9)+4))+5) )
		
		
		ifndef USE_BRGH
			ifndef USE_BRG16
				#define	UBRG		( (((FOSC / BAUDRATE) / 32) - 1) / 2 )	;baudrate
			else
				#define	UBRG		( (((FOSC / BAUDRATE) / 8) - 1) / 2 )	;baudrate
			endif
		endif
				
		ifdef USE_BRGH
			ifdef USE_BRG16
				#define	UBRG		( (((FOSC / BAUDRATE) / 2) - 1) / 2 )	;baudrate
			else
				#define	UBRG		( (((FOSC / BAUDRATE) / 8) - 1) / 2 )	;baudrate
			endif
		else
			
		endif
		
		#define	BLPLP		(BLPLW / PAGESIZEW)						;bootloader placement, pages from end
		#define	PAGESIZER	(PAGESIZEW/ROWSIZEW)					;pagesize [rows]
		#define	ROWSIZEB	(ROWSIZEW*2)							;rowsize [bytes]
		#define STARTADDR	( MAX_FLASH - BLPLP * PAGESIZER * ROWSIZEW )			;bootloader placement		

		; Debug output
		;messg	UBRG_IS #v(UBRG)
		;messg	bldelayis 	#v(BLDELAY)
		
		;
		errorlevel 			1, -305									; suppress warning msg that takes f as default
		
		
;------------------------------------------------------------------------------
; Extended PIC16F
;------------------------------------------------------------------------------
	ifndef EEDATL
		#define	EEDATL	EEDATA
	endif	
	ifndef EEADRL
		#define	EEADRL	EEADR
	endif
	ifndef INDF
		#define	INDF_	INDF0
	else
		#define	INDF_	INDF
	endif
	ifndef FSR
		#define	FSR_	FSR0L
	#ELSE
		#define	FSR_	FSR
	endif
	
	ifndef PMCON1
		#define	EECON1_		EECON1
		#define	EECON2_		EECON2
		#define	EEDATL_		EEDATL
		#define	EEDATH_		EEDATH
		#define	EEADRL_		EEADRL
		#define	EEADRH_		EEADRH
	else
		#define	EECON1_		PMCON1
		#define	EECON2_		PMCON2
		#define	EEDATL_		PMDATL
		#define	EEDATH_		PMDATH
		#define	EEADRL_		PMADRL
		#define	EEADRH_		PMADRH
	endif
	
	
;------------------------------------------------------------------------------
; UARTs
;------------------------------------------------------------------------------ 
		ifndef USE_UART1
			ifndef USE_UART2
				error "No communication is specified"
			endif
		endif
		
		ifdef USE_UART1
			ifdef USE_UART2
				error "Both uarts are specified"
			endif
			
			#define PIRRCIF_		PIR1				;uart interupt status register
			#define RCIF_			RCIF		
			
			ifdef TX1STA
			   	#define TXSTA_		TX1STA				;uart status
			   	#define	RCSTA_		RC1STA				;uart status
			   	#define	SPBRG_		SP1BRGL				;uart baudrate
			   	#define	SPBRGH_		SP1BRGH				;uart baudrate
			   	#define	TXREG_		TX1REG				;uart transmit
			   	#define	RCREG_		RC1REG				;uart receive
			   	#define	BAUDCON_	BAUD1CON			;uart receive
			else
			   	#define	TXSTA_		TXSTA				;uart status
			   	#define	RCSTA_		RCSTA				;uart status
			   	#define	SPBRG_		SPBRG				;uart baudrate
			   	#define	SPBRGH_		SPBRGH				;uart baudrate
			   	#define	TXREG_		TXREG				;uart transmit
			   	#define	RCREG_		RCREG				;uart receive
			   	#define	BAUDCON_	BAUDCON				;uart receive
			endif
		endif

		ifdef USE_UART2
			ifndef HAS_UART2
				error "UART2 specified for a device that only has uart1"
			endif			
			
			#define PIRRCIF_	PIR4					;uart interupt status register
			#define RCIF_		RC2IF

		   	#define	TXSTA_		TX2STA					;uart status
		   	#define	RCSTA_		RC2STA					;uart status
		   	#define	SPBRG_		SP2BRGL					;uart baudrate
			#define	SPBRGH_		SP2BRGH					;uart baudrate
		   	#define	TXREG_		TX2REG					;uart transmit
		   	#define	RCREG_		RC2REG					;uart receive
			#define	BAUDCON_	BAUD2CON				;uart receive
		endif


;------------------------------------------------------------------------------
; Variables
;------------------------------------------------------------------------------ 
		cblock 0x70		;0x70-0x7f are available in all banks
			crc			;receive checksum
			dcnt		;datacount
			cnt1		;receive timeout timer
			cnt2		;receive timeout timer
			cnt3		;receive timeout timer
			cntHello	;
			addrh		;high address byte
			cmd			;command
			doerase		;
			rowcnt
			rowcnt2
		endc
		cblock 0x20		;only bank 0 but accessed using indirect addressing, 0x20-0x70 = 80bytes
			buffer:(PAGESIZEW*2 + 1)	;receive buffer
		endc
		
		
;------------------------------------------------------------------------------
; Macros
;------------------------------------------------------------------------------ 	
SendL 	macro 	sbyte
			movlw 	sbyte
			call	Send
		endm

	
;------------------------------------------------------------------------------
; Reset vector
;------------------------------------------------------------------------------ 
		org     0x0000
		PAGESEL blstart
		goto    blstart


;------------------------------------------------------------------------------
; GOTO user application
;------------------------------------------------------------------------------ 	
		org 	STARTADDR - 3	;space to deposit goto to user application
loadusr	
		nop						;page select, replaced by gui
		nop						;page select, replaced by gui
		nop						;goto, replaced by gui
	
	
;------------------------------------------------------------------------------
; Start of bootloader code
;------------------------------------------------------------------------------ 	
		org 	STARTADDR
blstart	
	
		
;------------------------------------------------------------------------------
; User specific entry code go here
;------------------------------------------------------------------------------	

		; Set internal oscillator to 8MHz
		;banksel	OSCCON
		;bsf		OSCCON, IRCF0	;bank 1
		;bsf		OSCCON, IRCF1
		;bsf		OSCCON, IRCF2
		
		
;------------------------------------------------------------------------------
; Init
;------------------------------------------------------------------------------	
		
		
		
		;----------------------------------------------------------------------
		; TX enable, make tx enable pin output and set to 0
		;----------------------------------------------------------------------
		ifdef USE_TXENABLE
			banksel	TRISR_TXE
			bcf		TRISR_TXE, TRISB_TXE
			banksel	PORTR_TXE
			bcf		PORTR_TXE, PORTB_TXE
		endif		
		
		
		;----------------------------------------------------------------------
		; Init uart
		;---------------------------------------------------------------------- 		
		; Baud rate	
		banksel	SPBRG_
		movlw	(UBRG & 0xff)
		movwf	SPBRG_				;bank 1/3
		
		; Brg16
		ifdef USE_BRG16
			movlw	(UBRG >> 8)
			banksel	SPBRGH_
			movwf	SPBRGH_			;bank 3
			bsf		BAUDCON_, BRG16	;bank 3
		endif
		; Receive		
		movlw	b'10010000'
		banksel	RCSTA_
		movwf	RCSTA_				;bank 0/3
		; Transmit		
		ifdef USE_BRGH
			movlw	b'00100100'
		else
			movlw	b'00100000'
		endif
		banksel TXSTA_
		movwf	TXSTA_				;bank 1/3
		
		
		;----------------------------------------------------------------------
		; UART with auto baudrate detection
		;----------------------------------------------------------------------
		ifdef USE_ABAUD
			;Enable auto baud rate detection
			banksel BAUDCON_		;bank 3
			bsf		BAUDCON_, ABDEN			
			; Wait for auto baud rate detection to complete and do timeout control
			movlw 	BLDELAY
			movwf 	cnt1								
rpt2_		movlw	delay2
			movwf	cnt2		
rpt3_		clrf 	cnt3		
rptc_		clrwdt
			banksel	BAUDCON_
			btfss 	BAUDCON_, ABDEN	;bank 3
			goto 	abaudok
notrcv_		decfsz 	cnt3, f
			goto 	rptc_
			decfsz 	cnt2, f
			goto 	rpt3_
			decfsz 	cnt1, f
			goto 	rpt2_
			; Timed out if we get here			
			goto	exit
			;Send ok
abaudok		SendL 	OK				;->bank ?
		endif
		
		
;------------------------------------------------------------------------------
; Wait for computer
;------------------------------------------------------------------------------
waitpc	movlw	HELLOTRIES
		movwf	cntHello
rhello	call 	Receive			;->bank ?
		sublw 	HELLO
		skpnz	
		goto	sendid
		; Not hello received
		decfsz	cntHello
		goto	rhello	
		goto	exit
			
		
;------------------------------------------------------------------------------
; Send device id and firmware version
;------------------------------------------------------------------------------
sendid	SendL 	( DEVICEID & 0xff )	;->bank ?
		SendL	( ((DEVICEID&0x100)>>1) + VERMAJ )
		SendL	( ((DEVICEID&0x200)>>2) + (VERMIN<<4) + VERREV )
		
		
;------------------------------------------------------------------------------
; Main loop
;------------------------------------------------------------------------------	
Main	clrf	STATUS
		SendL 	OK				;->bank ?, everything OK, ready and waiting
mainl	clrf 	crc
		

		;----------------------------------------------------------------------
		; Receive address
		;----------------------------------------------------------------------			
		; Dummy upper byte
		call 	Receive			;->bank ?
		; High byte
		call 	Receive			;->bank ?
		banksel	EEADRH_			
		movwf 	EEADRH_			;bank 2/3
		movwf	addrh
		; Low byte				
		call 	Receive			;->bank ?
		banksel	EEADRL_
		movwf 	EEADRL_			;bank 2/3
		
		
		;----------------------------------------------------------------------
		; Receive command
		;----------------------------------------------------------------------			
		call 	Receive			;->bank ?	
		movwf 	cmd	
		
				
		;----------------------------------------------------------------------
		; Receive nr of data bytes that will follow
		;----------------------------------------------------------------------			
		call 	Receive			;->bank ?	
		movwf 	dcnt	
		
		
		;----------------------------------------------------------------------
		; Receive data
		;----------------------------------------------------------------------	
		movlw	buffer
		movwf	FSR_
rcvoct	call 	Receive			;->bank ?
		movwf 	INDF_
		incf	FSR_
		decfsz 	dcnt
		goto 	rcvoct
		
		
		;----------------------------------------------------------------------
		; Check checksum
		;----------------------------------------------------------------------			
		movf 	crc, f
		skpz
		goto 	crcfail
		
		
		;----------------------------------------------------------------------
		; Init buffer pointer
		;----------------------------------------------------------------------
		movlw	buffer
		movwf	FSR_			;available in all banks
		
				
		;----------------------------------------------------------------------
		; Check command
		;----------------------------------------------------------------------			
		; Erase page
		btfss	cmd, 0
		goto	cwrrow
		bsf		doerase, 0
		goto	Main		
		; Write row
cwrrow	btfsc 	cmd, 1			
		goto 	erase
		; Write eeprom word
		ifdef HAS_EE
			btfsc 	cmd, 2			
			goto	eeprom
		endif
		; Else, unknown command
		SendL	UCMD				;->bank?
		goto		mainl

		
		;----------------------------------------------------------------------
		; Erase page
		;----------------------------------------------------------------------					
erase	
		btfss 	doerase, 0
        goto    writerow		
		ifndef AUTOERASE
			movlw	0x94			;program mem, erase, enable write
			call 	Write			;->bank 3
			banksel	EECON1_
			bcf		EECON1_, FREE	;bank 2/3, disable erase
		endif
		clrf	doerase
		
						
		;----------------------------------------------------------------------
		; Write row
		;----------------------------------------------------------------------					
writerow
		movlw	ROWSIZEW
		movwf	rowcnt
		movwf	rowcnt2	
wrword
		; Load latches
		movfw 	INDF_
		banksel	EEDATL_
		movwf	EEDATL_			;bank 2/3
		incf	FSR_
		movfw	INDF_
		movwf	EEDATH_			;bank 2/3
		incf	FSR_
		;Write
		movlw	0x84			;program mem, enable write
		call	Write			;->bank 3
		; Word write complete, more words to be writen?
		banksel	EEADRL_		
		incf	EEADRL_			;bank 2/3
		decfsz	rowcnt
		goto	wrword

		;----------------------------------------------------------------------
		; Write complete, verify row
		;----------------------------------------------------------------------	
verword
		decf	EEADRL_			;bank 2/3
		; Read word
		banksel EECON1_		
		ifdef HAS_EE	
			bsf		EECON1_, EEPGD	;bank 3
		endif
		bsf		EECON1_, RD		;bank 3
		nop
		nop
		; Compare high byte	
		decf	FSR_
		movfw	INDF_
		;andlw	0x3F			;maybe we should only compare low 6 bits?
		banksel	EEDATH_
		subwf	EEDATH_			;bank 2/3
		btfss	STATUS, Z
		goto	verfail
		; Compare low byte			
		decf	FSR_
		movfw	INDF_
		subwf	EEDATL_			;bank 2/3
		btfss	STATUS, Z
		goto	verfail
		; Loop?
		decfsz 	rowcnt2
		goto 	verword	
		; Verify succesfull
		goto 	Main						
				
		
		;----------------------------------------------------------------------
		; Write eeprom byte
		;----------------------------------------------------------------------	
	ifdef HAS_EE			
		; Load latch
eeprom	movfw	INDF_
		banksel	EEDATL_
		movwf	EEDATL_			;bank 2/3
		; Write
		movlw 	0x04
		call 	Write			;->bank 3		
		; Wait for write to complete
eewait	
		btfsc	EECON1_, WR		;bank 3
		goto	eewait	
		
		
		;----------------------------------------------------------------------
		; Write complete, verify byte
		;----------------------------------------------------------------------				
		; Read
		bsf		EECON1_, RD		;bank 3
		movfw	INDF_
		; Compare
		banksel	EEDATL_
		subwf	EEDATL_			;bank 2/3
		btfss	STATUS, Z
		goto	verfail
		; Verify succesfull
		goto 	Main
	endif
				
				
		;----------------------------------------------------------------------
		; Verify fail
		;----------------------------------------------------------------------
verfail	
		SendL 	VERFAIL			;->bank ?
		goto 	mainl
		
						
		;----------------------------------------------------------------------
		; Checksum error
		;----------------------------------------------------------------------
crcfail	
		SendL 	CHECKSUMERR		;->bank ?
		goto 	mainl

		
;------------------------------------------------------------------------------
; Write()
;------------------------------------------------------------------------------		
Write
		;
		banksel	EECON1_
		movwf	EECON1_			;bank 3
		; Write		
		movlw	0x55
		movwf	EECON2_			;bank 3
		movlw	0xaa
		movwf	EECON2_			;bank 3
		bsf		EECON1_, WR		;bank 3, start write
		nop
		nop	
waitwre		
		btfsc	EECON1_, WR		;bank 3
		goto	waitwre				
		bcf		EECON1_, WREN	;bank 3 disable writes
		return
		
		
;------------------------------------------------------------------------------
; Send()
;------------------------------------------------------------------------------		
Send	; Enable tx
		ifdef USE_TXENABLE
			banksel	PORTR_TXE
			bsf	PORTR_TXE, PORTB_TXE
			nop		;needed?
			nop
			nop
		endif		
		;Send byte
		banksel	TXREG_
		movwf 	TXREG_			;bank 0/3
		; Wait for transmit shift register to get empty
		banksel	TXSTA_	
txwait	btfss	TXSTA_, TRMT	;bank 1/3
		goto	txwait
		; Disable tx 
		ifdef USE_TXENABLE
			banksel	PORTR_TXE
			bcf	PORTR_TXE, PORTB_TXE	;bank ?
		endif
		; Send complete
		return
		
		
;------------------------------------------------------------------------------
; Receive()
;------------------------------------------------------------------------------	
Receive	movlw 	BLDELAY
		movwf 	cnt1								
rpt2	movlw	delay2
		movwf	cnt2
		;clrf 	cnt2		
rpt3	clrf 	cnt3		
rptc	clrwdt
		banksel	PIRRCIF_
		btfss 	PIRRCIF_, RCIF_		;bank 0, test RX
		goto 	notrcv
		banksel RCREG_
		movfw 	RCREG_			;bank 0/3, return read data in W, same bank as PIR1
		addwf 	crc, f			;compute crc
		return		
notrcv	decfsz 	cnt3, f
		goto 	rptc
		decfsz 	cnt2, f
		goto 	rpt3
		decfsz 	cnt1, f
		goto 	rpt2
		; Receive timed out if we get here
		
		
;------------------------------------------------------------------------------
; Exit
;------------------------------------------------------------------------------			
exit	
		banksel	RCSTA_
		clrf	RCSTA_			;bank 0/3, reset receive status and control register
		banksel	TXSTA_
		clrf	TXSTA_			;bank 1/3, reset transmit status and control register
		goto 	loadusr
		
		
;------------------------------------------------------------------------------
; Functions
;------------------------------------------------------------------------------


;------------------------------------------------------------------------------
; End of code
;
; After reset
; Do not expect the memory to be zero,
; Do not expect registers to be initialised like in catalog.
;------------------------------------------------------------------------------			
		end
